<?php

if (!defined('ABSPATH')) {
    exit;
}

class OnePay_API {
    
    private $api_key;
    private $api_secret;
    private $api_url;
    private $test_mode;
    
    public function __construct() {
        $this->api_key = get_option('onepay_api_key', '');
        $this->api_secret = get_option('onepay_api_secret', '');
        $this->test_mode = get_option('onepay_test_mode', 'yes') === 'yes';
        $this->api_url = $this->test_mode 
            ? 'https://api.onepayltd.kz/test' 
            : 'https://api.onepayltd.kz/ac';
        
        OnePay_Logger::log('OnePay API Initialized');
        OnePay_Logger::log('Test Mode: ' . ($this->test_mode ? 'YES' : 'NO'));
        OnePay_Logger::log('API URL: ' . $this->api_url);
        OnePay_Logger::log('API Key: ' . (!empty($this->api_key) ? substr($this->api_key, 0, 8) . '...' . substr($this->api_key, -4) : 'NOT SET'));
    }
    
    private function get_headers() {
        return array(
            'Content-Type' => 'application/json',
            'api-key' => $this->api_key,
        );
    }
    
    private function sanitize_for_logging($data) {
        if (!is_array($data)) {
            return $data;
        }
        
        $sanitized = $data;
        $sensitive_fields = array('card', 'number', 'cvv', 'cvc', 'pan', 'password', 'secret', 'token');
        
        foreach ($sanitized as $key => $value) {
            $key_lower = strtolower($key);
            
            foreach ($sensitive_fields as $field) {
                if (strpos($key_lower, $field) !== false) {
                    $sanitized[$key] = '***REDACTED***';
                    continue 2;
                }
            }
            
            if (is_array($value)) {
                $sanitized[$key] = $this->sanitize_for_logging($value);
            }
        }
        
        return $sanitized;
    }
    
    public function create_payment($data) {
        $endpoint = '/payments';
        
        $payment_data = array(
            'amount' => $data['amount'],
            'currency' => $data['currency'],
            'payment_link'=> true,
            'return_url' => $data['return_url'],
            'payment_link_config' => array(
                'theme' => '#c61e00',
            ),
            'authentication_type' => 'three_ds',
        );
        
        // Add profile_id - required by OnePay API
        $profile_id = get_option('onepay_profile_id', '');
        if (!empty($profile_id)) {
            $payment_data['profile_id'] = $profile_id;
        } else {
            OnePay_Logger::log('Profile ID is not configured in OnePay settings. Please add your Profile ID (starts with "pro_") from OnePay dashboard.', 'error');
        }
        
        if (isset($data['customer_id']) && !empty($data['customer_id'])) {
            $customer_id = 'cus_' . $data['customer_id'];
        } elseif (isset($data['metadata']['customer_id']) && !empty($data['metadata']['customer_id'])) {
            $customer_id = 'cus_' . $data['metadata']['customer_id'];
        } else {
            $customer_id = 'cus_guest_' . time();
        }
        
        $payment_data['customer'] = array(
            'id' => $customer_id,
        );
        
        if (isset($data['customer_email']) && !empty($data['customer_email'])) {
            $payment_data['customer']['email'] = $data['customer_email'];
        }
        
        if (isset($data['customer_name']) && !empty($data['customer_name'])) {
            $payment_data['customer']['name'] = $data['customer_name'];
        }
        
        if (isset($data['description']) && !empty($data['description'])) {
            $payment_data['description'] = $data['description'];
        }
        
        if (isset($data['metadata']) && !empty($data['metadata'])) {
            $payment_data['metadata'] = $data['metadata'];
        }
        
        if (isset($data['order_details']) && !empty($data['order_details'])) {
            $payment_data['order_details'] = $data['order_details'];
        }
        
        if (isset($data['payment_method'])) {
            $payment_data['payment_method'] = $data['payment_method'];
        }
        
        return $this->make_request('POST', $endpoint, $payment_data);
    }
    
    public function capture_payment($payment_id, $amount = null) {
        $endpoint = '/payments/' . $payment_id . '/capture';
        
        $data = array();
        if ($amount !== null) {
            $data['amount_to_capture'] = $amount;
        }
        
        return $this->make_request('POST', $endpoint, $data);
    }
    
    public function create_subscription($data) {
        $endpoint = '/subscriptions';
        
        $subscription_data = array(
            'customer' => array(
                'email' => $data['customer_email'],
                'name' => $data['customer_name'],
            ),
            'plan_id' => $data['plan_id'],
            'payment_method' => $data['payment_method'],
            'metadata' => isset($data['metadata']) ? $data['metadata'] : array(),
        );
        
        return $this->make_request('POST', $endpoint, $subscription_data);
    }
    
    public function cancel_subscription($subscription_id) {
        $endpoint = '/subscriptions/' . $subscription_id . '/cancel';
        return $this->make_request('POST', $endpoint, array());
    }
    
    public function create_refund($payment_id, $amount = null, $reason = '') {
        $endpoint = '/refunds';
        
        $refund_data = array(
            'payment_id' => $payment_id,
            'reason' => $reason,
        );
        
        if ($amount !== null) {
            $refund_data['amount'] = $amount;
        }
        
        return $this->make_request('POST', $endpoint, $refund_data);
    }
    
    public function get_payment($payment_id) {
        $endpoint = '/payments/' . $payment_id;
        return $this->make_request('GET', $endpoint);
    }
    
    public function get_refund($refund_id) {
        $endpoint = '/refunds/' . $refund_id;
        return $this->make_request('GET', $endpoint);
    }
    
    public function create_customer($data) {
        $endpoint = '/customers';
        
        $customer_data = array(
            'email' => $data['email'],
            'name' => $data['name'],
            'phone' => isset($data['phone']) ? $data['phone'] : '',
            'metadata' => isset($data['metadata']) ? $data['metadata'] : array(),
        );
        
        return $this->make_request('POST', $endpoint, $customer_data);
    }
    
    public function create_payment_method($data) {
        $endpoint = '/payment_methods';
        
        $payment_method_data = array(
            'type' => $data['type'],
            'card' => isset($data['card']) ? $data['card'] : array(),
            'billing_details' => isset($data['billing_details']) ? $data['billing_details'] : array(),
        );
        
        return $this->make_request('POST', $endpoint, $payment_method_data);
    }
    
    private function make_request($method, $endpoint, $data = array()) {
        $url = $this->api_url . $endpoint;
        
        $args = array(
            'method' => $method,
            'headers' => $this->get_headers(),
            'timeout' => 30,
        );
        
        if (!empty($data) && in_array($method, array('POST', 'PUT', 'PATCH'))) {
            $args['body'] = json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
            OnePay_Logger::log('JSON Encoding: Using JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES flags for Cyrillic support');
        }
        
        OnePay_Logger::log('API Request: ' . $method . ' ' . $url);
        OnePay_Logger::log('Request Headers: api-key=' . (!empty($this->api_key) ? substr($this->api_key, 0, 8) . '...' . substr($this->api_key, -4) : 'NOT SET'));
        OnePay_Logger::log('Request Data: ' . print_r($this->sanitize_for_logging($data), true));
        
        $response = wp_remote_request($url, $args);
        
        if (is_wp_error($response)) {
            OnePay_Logger::log('API Error: ' . $response->get_error_message(), 'error');
            return array(
                'success' => false,
                'error' => $response->get_error_message(),
            );
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        $response_data = json_decode($response_body, true);
        
        OnePay_Logger::log('API Response Code: ' . $response_code);
        OnePay_Logger::log('API Response: ' . print_r($this->sanitize_for_logging($response_data), true));
        
        if ($response_code >= 200 && $response_code < 300) {
            return array(
                'success' => true,
                'data' => $response_data,
            );
        } else {
            $error_message = 'Unknown error';
            
            if (isset($response_data['error'])) {
                if (is_string($response_data['error'])) {
                    $error_message = $response_data['error'];
                } elseif (is_array($response_data['error'])) {
                    $error_message = isset($response_data['error']['message']) 
                        ? $response_data['error']['message'] 
                        : json_encode($response_data['error']);
                }
            } elseif (isset($response_data['message'])) {
                $error_message = $response_data['message'];
            } elseif (!empty($response_body)) {
                $error_message = 'HTTP ' . $response_code . ': ' . substr($response_body, 0, 200);
            }
            
            OnePay_Logger::log('Payment Error: ' . $error_message, 'error');
            
            return array(
                'success' => false,
                'error' => $error_message,
                'code' => $response_code,
                'data' => $response_data,
            );
        }
    }
}
